/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datasource;

import com.mongodb.ConnectionString;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.errors.OlapBuilderErrorCode;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.errors.OlapBuilderErrorManager;
import crazydev.iccube.builder.model.def.IOlapBuilderDataSource;
import crazydev.iccube.builder.model.validation.OlapBuilderBaseDataSourceValidator;
import crazydev.iccube.builder.model.validation.OlapBuilderValidationContext;
import crazydev.iccube.builder.mongodb.common.OlapBuilderMongoDbHelper;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.Nullable;

public class OlapBuilderMongoDbDataSourceValidator extends OlapBuilderBaseDataSourceValidator
{
    public OlapBuilderMongoDbDataSourceValidator()
    {
    }

    @Override
    protected void doValidateFields(OlapBuilderValidationContext context, IOlapBuilderDataSource dataSource)
    {
        super.doValidateFields(context, dataSource);

        final OlapBuilderMongoDbDataSource myDataSource = (OlapBuilderMongoDbDataSource) dataSource;

        validateNotBlankField(context, OlapBuilderMongoDbDataSource.PASSWORD, myDataSource.getDbName());

        final String connectionString = myDataSource.getDecodedConnectionString();
        if (CdStringUtils.isNotNullAndNotBlank(connectionString))
        {
            validateConnectionString(context.getErrorManager(), connectionString);
        }
        else
        {
            validateNotBlankField(context, OlapBuilderMongoDbDataSource.SERVER_NAME, myDataSource.getServerName());
            validateNotBlankField(context, OlapBuilderMongoDbDataSource.DB_NAME, myDataSource.getDbName());

            try
            {
                ((OlapBuilderMongoDbDataSource) dataSource).getDateTimeTZ() /* side effect : invalid is throwing a builder error */;
            }
            catch (OlapBuilderErrorException ex)
            {
                context.addError(OlapBuilderMongoDbDataSource.DATE_TIME_TZ, OlapBuilderErrorCode.TIMEZONE_INVALID, OlapBuilderMongoDbDataSource.DATE_TIME_TZ.getName());

            }
            OlapBuilderMongoDbHelper.validate(myDataSource.getProperties());
        }

    }

    static void validateConnectionString(OlapBuilderErrorManager errorManager, @Nullable String connectionString)
    {
        try
        {
            if (connectionString == null)
            {
                errorManager.addError(OlapBuilderMongoDbDataSource.CONNECTIONSTRING, OlapBuilderErrorCode.FIELD_IS_NULL, OlapBuilderMongoDbDataSource.CONNECTIONSTRING.getName());
                return;
            }
            if (StringUtils.isBlank(connectionString))
            {
                errorManager.addError(OlapBuilderMongoDbDataSource.CONNECTIONSTRING, OlapBuilderErrorCode.FIELD_IS_EMPTY, OlapBuilderMongoDbDataSource.CONNECTIONSTRING.getName());
                return;
            }
            new ConnectionString(connectionString);
        }
        catch (RuntimeException ex)
        {
            errorManager.addError(OlapBuilderMongoDbDataSource.CONNECTIONSTRING, OlapBuilderErrorCode.UNEXPECTED_ERROR, ex.getLocalizedMessage());
        }
    }


}
