/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datasource;

import com.mongodb.*;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.MongoIterable;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.datasource.OlapBuilderAbstractConnection;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.mongodb.error.OlapBuilderMongoDbErrorCode;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class OlapBuilderMongoDbConnection extends OlapBuilderAbstractConnection<OlapBuilderMongoDbDataSource>
{
    private com.mongodb.client.MongoClient mongoClient;

    @Nullable
    private MongoDatabase mongoDatabase;

    public OlapBuilderMongoDbConnection(OlapBuilderMongoDbDataSource dataSource)
    {
        super(dataSource);
    }

    @Nullable
    private static char[] password(@Nullable String password)
    {
        return (password != null) ? password.toCharArray() : null;
    }

    public MongoClient getMongoClient()
    {
        return mongoClient;
    }

    public MongoDatabase getMongoDataBase()
    {
        if (mongoDatabase == null)
        {
            mongoDatabase = mongoClient.getDatabase(dataSource.getDbName());
        }
        return mongoDatabase;
    }

    @Override
    protected void checkOnOpenConnectionEx() throws OlapBuilderErrorException
    {
        // Having a mongoDB is not enough for an actual connection to the server.
        // Let's retrieve some general information...

        try
        {
            final Bson buildInfo = new BasicDBObject("buildInfo", 1);
            final Document commandResult = getMongoDataBase().runCommand(buildInfo);
            final String serverVersion = commandResult.getString("version");
        }
        catch (MongoException ex)
        {
            OlapLoggers.BUILDER.error("MongoDB connection error", ex);
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, ex.getMessage());
        }

    }

    @Override
    protected void onOpen(@Nullable OlapRuntimeContext context) throws OlapBuilderErrorException
    {
        mongoClient = createMongoClient(-1);
    }

    private com.mongodb.client.MongoClient createMongoClient(int serverSelectionTimeout)
    {
        final OlapBuilderMongoDbDataSource ds = getDataSource();

        final String database = ds.getDbName();

        // Credentials are not required in ' secure mode '.

        String userName = ds.getUser();

        if (CdStringUtils.isNullOrBlank(userName))
        {
            userName = null;
        }

        String password = ds.getPassword();

        if (CdStringUtils.isNullOrBlank(password))
        {
            password = null;
        }

        try
        {
            final MongoClient mongoClient;
            String connectionString = ds.getDecodedConnectionString();
            if (CdStringUtils.isNotNullAndNotBlank(connectionString))
            {
                mongoClient = MongoClients.create(connectionString);
            }
            else
            {
                mongoClient = buildMongoClient(ds, database, userName, password, serverSelectionTimeout);
            }
            return mongoClient;
        }
        catch (MongoException ex)
        {
            OlapLoggers.BUILDER.error("MongoDB connection error", ex);
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, ex.getMessage());
        }
    }

    @NotNull
    private MongoClient buildMongoClient(OlapBuilderMongoDbDataSource ds, String database, String userName, String password, int serverSelectionTimeout)
    {
        MongoClientSettings.Builder clientBuilder = ds.getOptions(MongoClientSettings.builder(), serverSelectionTimeout);

        final MongoCredential credential;
        if (userName != null)
        {
            credential = getCredentials(ds, database, userName, password);
        }
        else
        {
            credential = null;
        }

        // =========================================================================================================
        // Read preference describes how MongoDB clients route read operations to members of a replica set.
        //
        // By default, an application directs its read operations to the primary member in a replica set.
        // Reading from the primary guarantees that read operations reflect the latest version of a document.
        // However, by distributing some or all reads to secondary members of the replica set, you can improve
        // read throughput or reduce latency for an application that does not require fully up-to-date data.
        //
        // For now I guess PRIMARY is fine for our usage (always read fully up-to-date data).
        // =========================================================================================================

        {
            if ("snappy".equalsIgnoreCase(ds.getCompression()))
            {
                clientBuilder = clientBuilder.compressorList(Collections.singletonList(
                        MongoCompressor.createSnappyCompressor()
                ));
            }
            if ("zlib".equalsIgnoreCase(ds.getCompression()))
            {
                clientBuilder = clientBuilder.compressorList(Collections.singletonList(
                        MongoCompressor.createZlibCompressor()
                ));
            }
        }

        if (credential != null)
        {
            clientBuilder.credential(credential);
        }
        final String connectionString = "mongodb://" + new ServerAddress(ds.getServerName(), ds.getPortNumber()).toString();
        final MongoClientSettings clientSettings = clientBuilder
                .applyConnectionString(new ConnectionString(connectionString))
                .build();

        return MongoClients.create(clientSettings);
    }

    @SuppressWarnings("deprecation")
    private MongoCredential getCredentials(OlapBuilderMongoDbDataSource ds, String database, String userName, String password)
    {
        final OlapBuilderMongoDbAuth authMechanism = ds.getAuthMechanism();
        switch (authMechanism)
        {
            case DEFAULT:
                return MongoCredential.createCredential(userName, database, password(password));
            case SCRAM_SHA_1:
                return MongoCredential.createScramSha1Credential(userName, database, password(password));
            case CR:
                return MongoCredential.createMongoCRCredential(userName, database, password(password));
            case X509:
                return MongoCredential.createMongoX509Credential(userName);
            case GSSAPI:
                return MongoCredential.createGSSAPICredential(userName);
            case PLAIN:
                return MongoCredential.createPlainCredential(userName, database, password(password));
            default:
                throw new RuntimeException("internal error: unexpected authentication [" + authMechanism.name() + "]");
        }
    }

    @Override
    protected void onClose()
    {
        if (mongoClient != null)
        {
            mongoClient.close();
        }
    }

    public List<String> getCollectionNames()
    {
        // this is a best effort don't spend more than one sec.
        final MongoIterable<String> iter = mongoClient.getDatabase(dataSource.getDbName()).listCollectionNames();
        List<String> names = new ArrayList<>();
        iter.forEach((Block<String>) names::add);
        return names;
    }
}

