/*
 * Copyright 2014 - 2018 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package com.iccube.bson;

import com.iccube.bson.decoder.*;
import org.jetbrains.annotations.NotNull;

public enum ic3BsonType
{
    /**
     * Not a real BSON type. Used to signal the end of a document.
     */
    END_OF_DOCUMENT(0x00, new ic3BsonEndDocumentDecoder()),
    /**
     * A BSON double.
     */
    DOUBLE(0x01, new ic3BsonDoubleDecoder()),
    /**
     * A BSON string.
     */
    STRING(0x02, new ic3BsonStringDecoder()),
    /**
     * A BSON document.
     */
    DOCUMENT(0x03, new ic3BsonDocumentDecoder()),
    /**
     * A BSON array.
     */
    ARRAY(0x04, new ic3BsonArrayDecoder()),
    /**
     * BSON binary data.
     */
    BINARY(0x05, new ic3BsonBinary()),
    /**
     * A BSON undefined value.
     */
    UNDEFINED(0x06, new ic3BsonEmptyDecoder()),
    /**
     * A BSON ObjectId.
     */
    OBJECT_ID(0x07, new ic3BsonObjectIdDecoder()),
    /**
     * A BSON bool.
     */
    BOOLEAN(0x08, new ic3BsonBooleanDecoder()),
    /**
     * A BSON DateTime.
     */
    DATE_TIME(0x09, new ic3BsonDateTimeDecoder()),
    /**
     * A BSON null value.
     */
    NULL(0x0a, new ic3BsonNullDecoder()),
    /**
     * A BSON regular expression.
     */
    REGULAR_EXPRESSION(0x0b, new ic3BsonRegularExpressionDecoder()),
    /**
     * A BSON regular expression.
     */
    DB_POINTER(0x0c, new ic3BsonStringDecoder()),
    /**
     * BSON JavaScript code.
     */
    JAVASCRIPT(0x0d, new ic3BsonStringDecoder()),
    /**
     * A BSON symbol.
     */
    SYMBOL(0x0e, new ic3BsonStringDecoder()),
    /**
     * BSON JavaScript code with a scope (a set of variables with values).
     */
    JAVASCRIPT_WITH_SCOPE(0x0f, new ic3BsonJavascriptWithScopeDecoder()),
    /**
     * A BSON 32-bit integer.
     */
    INT32(0x10, new ic3BsonIntDecoder()),
    /**
     * A BSON timestamp.
     */
    TIMESTAMP(0x11, new ic3BsonTimeStampDecoder()),
    /**
     * A BSON 64-bit integer.
     */
    INT64(0x12, new ic3BsonLongDecoder()),
    /**
     * A BSON Decimal128.
     *
     * @since 3.4
     */
    DECIMAL128(0x13, new ic3BsonDecimal128Decoder()),
    /**
     * A BSON MinKey value.
     */
    MIN_KEY(0xff, new ic3BsonEmptyDecoder()),
    /**
     * A BSON MaxKey value.
     */
    MAX_KEY(0x7f, new ic3BsonEmptyDecoder());

    private static final ic3BsonType[] LOOKUP_TABLE = new ic3BsonType[MIN_KEY.getValue() + 1];

    static
    {
        for (final ic3BsonType cur : ic3BsonType.values())
        {
            LOOKUP_TABLE[cur.getValue()] = cur;
        }
    }

    private final int value;

    private final ic3BsonDecoder decoder;

    ic3BsonType(final int value, ic3BsonDecoder decoder)
    {
        this.value = value;
        this.decoder = decoder;
        if (decoder == null)
        {
            throw new RuntimeException("No decoder for " + Integer.toHexString(value));
        }
    }

    /**
     * Gets the {@code BsonType} that corresponds to the given int value.
     *
     * @param value the int value of the desired BSON type.
     *
     * @return the corresponding {@code BsonType}
     */
    @NotNull
    public static ic3BsonType findExistingType(final int value)
    {
        ic3BsonType type = LOOKUP_TABLE[value & 0xFF];
        if (type == null)
        {
            throw new RuntimeException("Type not found " + Integer.toHexString(value));
        }
        return type;
    }

    /**
     * Get the int value of this BSON type.
     *
     * @return the int value of this type.
     */
    int getValue()
    {
        return value;
    }

    /**
     * Returns whether this type is some sort of containing type, e.g. a document or array.
     *
     * @return true if this is some sort of containing type rather than a primitive value
     */
    public boolean isContainer()
    {
        return this == DOCUMENT || this == ARRAY;
    }

    public ic3BsonDecoder getDecoder()
    {
        return decoder;
    }
}
