/*
 * Copyright 2014 - 2018 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package com.iccube.bson;

import org.bson.ByteBuf;
import org.bson.io.ByteBufferBsonInput;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

class ic3BsonStringBuilder
{
    @Nullable
    private final byte[] bytes;

    private final int size /* size contains the \0 */;

    private final int hash;

    private int start;

    private ic3BsonStringBuilder(byte[] bytes, int start, int size, int hash)
    {
        this.start = start;
        this.bytes = bytes;

        this.size = size;
        this.hash = hash;
    }

    static StringAndLength getOrCreateString(Map<ic3BsonStringBuilder, String> cache, byte[] buffer, int start)
    {
        int hash = 1;

        byte bb;
        int pos = start;
        while ((bb = buffer[pos++]) != 0)
        {

            hash = 31 * hash + bb;

        }
        final int size = pos - start - 1;
        if (size == 1)
        {

            final byte asciiByte = buffer[start];
            if (asciiByte < 0)
            {
                return new StringAndLength(ByteBufferBsonInput.UTF8_CHARSET.newDecoder().replacement(), 2);
            }

            return new StringAndLength(ByteBufferBsonInput.ONE_BYTE_ASCII_STRINGS[asciiByte], 2);
        }
        else
        {
            final ic3BsonStringBuilder cStr = new ic3BsonStringBuilder(buffer, start, size, hash);
            final String str = cache.computeIfAbsent(cStr, key ->
                    new String(buffer, start, size, ByteBufferBsonInput.UTF8_CHARSET)
            );
            return new StringAndLength(str, size);
        }
    }

    @Override
    public int hashCode()
    {

        return hash;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (obj == null || getClass() != obj.getClass())
        {

            return false;

        }

        final ic3BsonStringBuilder obj_ = (ic3BsonStringBuilder) obj;

        if (size != obj_.size)
        {

            return false;
        }
        if (bytes != null && obj_.bytes != null)
        {
            for (int i = 0; i < size; i++)
            {
                if (bytes[start + i] != obj_.bytes[obj_.start + i])
                {
                    return false;
                }
            }
            return true;
        }

        throw new RuntimeException("internal error");
    }

    private boolean equals(ByteBuf buffer, byte[] bytes)
    {

        final int mark = buffer.position() - bytes.length;

        try
        {

            buffer.position(mark) /* restore for read */;

            for (int ii = 0; ii < bytes.length; ii++)
            {

                if (buffer.get() != bytes[ii])
                {
                    return false;
                }
            }

            return true;
        }
        finally
        {

            buffer.position(mark + bytes.length);

            // *** the buffer has consumed the c_str *******************************************************************
        }
    }

    static final class StringAndLength
    {
        final String str;

        final int bytesLength;

        StringAndLength(String str, int bytesLength)
        {
            this.bytesLength = bytesLength;
            this.str = str;
        }
    }
}
