## MDX Histograms

This article describes a dynamic histogram, you might read
[this post](http://stackoverflow.com/questions/31292009/histrogram-in-mdx-with-iccube)
in stackoverflow for an alternative solution that uses a previously defined hierarchy.

Histograms can be a straightforward job in MDX if you want to report on existing dimensions and hierarchies.
The situation changes if you're interested in doing a histogram on a variable that cannot be mapped into an
existing dimension. We're going to show how thanks to icCube OO capabilities doing histogram is
straightforward in any scenario.

We're going to use a [vector](../../mdx_types/vector.md) object as a base for our query that is going to compute
the distribution of daily sales in our `Sales` demo schema. The series of data can be calculated as following :

```
Vector( [Time].[Calendar].[Day].members , [Measures].[Amount]  )
```

The vector type supports a list of [methods](../../mdx_types/vector.md) and more specifically `hist(end,start)`
and `phist(end,start)` that we're going to use in this example. By default, the vector function includes empty
values as 0. In our example this means that days without sales are going to be included as 0 to the vector.
In the case we would only be interested in performing an analysis on the days with a sale we would use the
`EXCLUDEEMPTY` flag of the vector function.

Now that we have these two bricks we can now start building our MDX request :

```
WITH
    FUNCTION salesDistr() AS vector( [Time].[Calendar].[Day].members , [Measures].[Amount]  )
    
    MEMBER [Measures].[0]    AS salesDistr()->phist(NULL,1), FORMAT_STRING='percent'
    MEMBER [Measures].[1000] AS salesDistr()->phist(1,1000), FORMAT_STRING='percent'
    
SELECT
    { [0], [1000] } on 0
FROM
    [Sales]
```

We start by declaring a function that calculates our list of daily sales. In our calculated measures `[0]`
and `[1000]` we just have to call the `phist()` function to calculate the percentage of values between the
specified ranges. This example is easy to understand, but has a performance drawback as the list of sales
is calculated for each calculated measure (actually for each function call).

To solve this we have two different solutions. The first one is declaring our function constant :

```
WITH
    CONST FUNCTION salesDistr() AS vector( [Time].[Calendar].[Day].members , [Measures].[Amount]  )

    MEMBER [Measures].[0]    AS salesDistr()->phist(NULL,1), FORMAT_STRING='percent'
    MEMBER [Measures].[1000] AS salesDistr()->phist(1,1000), FORMAT_STRING='percent'
SELECT
    { [0], [1000] } on 0
FROM
    [Sales]
```

The drawback of this solution is that we can have a single `salesDistr()` per request and the query will not work
as expected once adding a new axis (e.g., to have several distributions by continent or sales people). To solve
this we can use an intermediate calculated member that might be cached. Assuming the icCube server cache mode
is `ON_DEMAND`, the `REQUEST_CACHED` cell property is set to true. This query looks like :

```
WITH
    MEMBER [Stats].[Stats-Dim].[distr] AS vector( [Time].[Calendar].[Day].members , ([Stats].[Stats-Dim].defaultMember, [Measures].[Amount] )), FORMAT_STRING='percent', REQUEST_CACHED = true
    
    -- currentTupleValue() is equivalent Value( [Measures].defaultmember )
    MEMBER [Measures].[0]    AS currentTupleValue()->phist(NULL,1), FORMAT_STRING='percent'
    MEMBER [Measures].[1000] AS currentTupleValue()->phist(1,1000), FORMAT_STRING='percent'

    SELECT
          { [0],[1000] } on 0
    FROM
      [Sales]
    WHERE ( [Stats].[Stats-Dim].[distr] )
```    

Note we use `currentTupleValue()` to retrieve the value of the cell (i.e., the vector). Eventually the full
request will look like :

```
WITH
    MEMBER [distr] AS vector( [Time].[Calendar].[Day].members, ([Stats].[Stats-Dim].defaultMember, [Measures].[Amount] ), EXCLUDEEMPTY ), FORMAT_STRING='percent', REQUEST_CACHED = true

    MEMBER     [0] AS Value([distr])->phist(NULL,     1), FORMAT_STRING='percent'
    MEMBER  [1000] AS Value([distr])->phist(   1,  1000), FORMAT_STRING='percent'
    MEMBER  [2000] AS Value([distr])->phist(1000,  2000), FORMAT_STRING='percent'
    MEMBER  [3000] AS Value([distr])->phist(2000,  3000), FORMAT_STRING='percent'
    MEMBER  [4000] AS Value([distr])->phist(3000,  4000), FORMAT_STRING='percent'
    MEMBER  [5000] AS Value([distr])->phist(4000,  5000), FORMAT_STRING='percent'
    MEMBER  [6000] AS Value([distr])->phist(5000,  6000), FORMAT_STRING='percent'
    MEMBER  [7000] AS Value([distr])->phist(6000,  7000), FORMAT_STRING='percent'
    MEMBER  [8000] AS Value([distr])->phist(7000,  8000), FORMAT_STRING='percent'
    MEMBER  [9000] AS Value([distr])->phist(8000,  9000), FORMAT_STRING='percent'
    MEMBER [10000] AS Value([distr])->phist(9000, 10000), FORMAT_STRING='percent'
    MEMBER [11000] AS Value([distr])->phist(10000,11000), FORMAT_STRING='percent'
    MEMBER [12000] AS Value([distr])->phist(11000, NULL), FORMAT_STRING='percent'
    
SELECT
    [Measures].allMembers - StripCalculatedMembers( [Measures].allMembers ) - [distr] on 0
FROM [Sales]
```

Next chapter: [MDX Categories](categories.md) describes how to define categories in MDX.

_