## Embedding icCube: JWT-Token (Session-Less)

For this scenario, we assume the host application is _session-less_ and uses a `JWT-Token` to pass around
user's claims from which the icCube authorization is built upon. This scenario is not strictly limited to `JWT-Tokens`.
The most important part is that it is not based on a session, meaning it requires additional configuration to match
icCube's requirements. Also, you can choose to pass the authorization information using an alternate mechanism.

This section assumes basic knowledge of icCube. You can refer to this [page](../Embedded.md) for an overview
about embedding icCube within a host Web application.

### Authorization Headers

By default, the icCube server does not know how to decode JWT-Token claims. Similarly to the scenario described
[here](./EmbeddedCookie.md), you will have to configure it to retrieve authentication/authorization information via
HTTP headers instead of the login form. You will have to change your `icCube.xml` for that. More on this later
in this document.

Once the host application authenticated the user successfully, it should send dedicated HTTP headers to icCube.
This is typically done in the reverse proxy (or in any other service) of the host applications backend. There,
the host application backend should be able to retrieve from the `JWT-Token` claims the information required
by icCube (e.g., the name of a role).

![Overview](./images/embedded-jwt-token.png)

**Reverse Proxy**

Even if strongly advised, the reverse proxy is not mandatory. Any other service of the backend could do the trick
of converting the `JWT-Token` into HTTP headers. Then, this other service is actually going to act as a reverse proxy for
icCube.

At the end of this document, we provide an example Express application that functions as the reverse proxy.

**icCube Ajax Request & JWT-Token**

The difference from the [cookie session-oriented](./EmbeddedCookie.md) architecture is that the icCube Javascript
code being embedded into the host application in the browser needs to explicitely send the `JWT-Token` on each request.
We explain this more and show an example later in this document.

**Request Flow**

The following picture shows a quick overview of the HTTP request flow from the Web browser to icCube.
The backend reverse proxy logic is mainly responsible to convert the `JWT-Token` claims into icCube dedicated
HTTP headers (i.e., user and role names as configured in this document).

![Overview](./images/embedded-jwt-token-flow.png)

### icCube Server Configuration (icCube.xml)

You can configure the icCube server with the `bin/icCube.xml` file. At the end of this section you can download a completed `icCube.xml` with all the needed settings.

The default servlet filters are responsible to display a `FORM` for unauthenticated users. To change this, you replace this with a so called `PASSTHROUGH` filter. This filter delegates the hard work to the header authentication
service:

```xml

<filter>
    <filter-name>PASSTHROUGH</filter-name>
    <filter-class>
        crazydev.iccube.server.authentication.passthrough.IcCubePassthroughAuthenticationServletFilter
    </filter-class>
    <init-param>
        <param-name>anonymousLogon</param-name>
        <param-value>false</param-value>
    </init-param>
    <init-param>
        <param-name>sessionLess</param-name>
        <param-value>true</param-value>
    </init-param>
    <init-param>
        <param-name>publicAssetsRole</param-name>
        <param-value>standard</param-value>
    </init-param>
</filter>
```

The `sessionLess` parameter above instructs icCube to invalidate any HTTP session it creates. Additionally, in icCube.xml, setting `webApp.sessionLess` to `true`  prevents icCube from generating any `JSESSIONID` cookies. These settings together make icCube session-less.

The `publicAssetsRole` references an existing icCube role used to retrieve the HTTP assets (e.g., HTML, JS, CSS,
etc...). Indeed, the HTTP requests sent by the icCube code embedded in the host application in the browser cannot
send the `JWT-Token`, and therefore won't have the required icCube HTTP headers. So this role must give access to
the `Docs` in read-only mode. The following URLs contain the `JWT-Token`:

```
/icCube/doc/* (except /icCube/doc/ic3-reporting/app/*)
/icCube/report/ic3-reporting/app-local/*
/icCube/ux-api/*
/icCube/api/*
/icCube/gvi/*
```

Note that the `CSRF` mitigation has been switched off as this should not be required in this scenario and would
not work in most cases because of the session-less architecture.

Also, all references to `anonymous` login are removed.

The default authentication service uses users and roles as defined within icCube. With the `PASSTHROUGH` filter,
it is replaced by the `IcCubeHeaderAuthenticationService` that uses dedicated HTTP headers to retrieve the name
of the authenticated user, and the information required to set up the role.

```xml

<icCubeAuthenticationService>

    <service-class>crazydev.iccube.server.authentication.IcCubeHeaderAuthenticationService</service-class>

    <param>
        <name>header.user</name>
        <value>IC3_USER_NAME</value>
    </param>

    <param>
        <name>header.role</name>
        <value>IC3_ROLE_NAME</value>
    </param>

    <param>
        <name>header.role.onTheFly</name>
        <value>false</value>
    </param>

</icCubeAuthenticationService>
```

The above configures the `icCubeAuthenticationService` service to use the HTTP headers `IC3_USER_NAME` and `IC3_ROLE_NAME`. The role send to
icCube in the HTTP header must exist in icCube. For more complex scenarios, where the actual content of the role is
defined on-the-fly (e.g., passing the name of a schema to authorize), please refer to this
[page](../OnTheFlyAuthorization.md) for more details.

See [icCube.xml](./icCube_jwt_token.xml) for a fully configured file. You can check the actual differences
with your own `icCube.xml` file for a better understanding of the changes.

### Client (Javascript)

The Javascript code of the host application has to be extended to embed icCube dashboards and/or the administration
interface.

**Dashboards**

The simplest integration is to add an `iframe` with permalinks for each dashboard you want to show. If you want more
control over icCube (i.e., open dashboard, sent events to dashboards, react from dashboard events, etc…), you can use
the icCube [API](https://github.com/ic3-software/ic3-reporting-api-embedded/blob/main/doc/Overview.md).

**Administration Interface**

The Server UI (i.e., both the admin and schema builder interfaces) can be embedded using an `iframe` with the
URL `/icCube/console/` as its `src` attribute. Note that since icCube 9.x, this URL is not limited anymore to
`/icCube/console/` and can for example be `/dashboards-api/console/` instead to better support your own reverse proxy logic.

Add the URL parameter `ic3configuration` ([www](../../dashboards/api/embed/EmbeddingConfiguration.md)) to instruct
icCube
the application is being embedded into a host application. The user interface is then configured accordingly (e.g.,
removing/replacing open in new tab, logout, etc...).

**JWT-Token / Custom HTTP Headers**

In this scenario, the icCube code must be configured to request the host application the value of the `JWT-Token`.
After the front-end loads the icCube libraries, icCube requests the HTTP headers that it adds to the outgoing AJAX
requests (the ones that needs to be authenticated).

This is done by adding the `?ic3customHeaders=xyz` parameter to the `iframe` URL. There is a similar parameter
if you're using the icCube API instead. You can use any value (see the `ic3-custom-headers-request` message below).

Once configured with this parameter, the icCube Javascript code first requests the host application what headers to add to each request. It achieves this by posting a message event. The answer to this message is the place where you set the value of the `JWT-Token`.
The host application should be extended to contain the following:

```javascript
// Listen to the post message send by the icCube JS libraries.
// The `type` of this message is `ic3-custom-headers-request`.
window.addEventListener("message", event => {

    const data = event.data;

    if (data.type === "ic3-custom-headers-request") {

        const embeddedDiv = (data.ic3callerType === "div");
        const ic3customheaders = data.ic3customheaders /* as specified in the URL */;
        
        // Post the reply to the window when embedding as DIV and to the iFrame otherwise.
        const target = !embeddedDiv
            ? document.getElementById("ic3-iframe")?.["contentWindow"]
            : window
        ;

        target && target.postMessage(
            {
                type: "ic3-custom-headers-reply",
                data: {
                    headers: {
                        "Authorization": "JWT @xyz",
                    }
                }
            },
            "*"
        );
    }
})
```

The message `ic3-custom-headers-request` contains the following fields:

```
data.ic3callerType    : "div" | "iframe"
data.ic3customHeaders : the "xyz" value used in the ic3customHeaders parameter
```

This handler is responsible to post back a message that contains the HTTP headers to be added to each request.
Notice how the handler is switching the target. You should use here the correct `iframe` ID (either hardcoded
or possibly coming from the `ic3customHeader` parameter).

Note that if the host application is **refreshing** the `JWT-Token`, it should post the new value. icCube adds 
such a listener for the life-cycle of the application and uses the latest value it received.

You can search for the `withCustomHeaders` in the source code of the live example mentioned later in this document.

### Troubleshooting

In icCube's `bin/log4j.xml` file, you can configure several loggers helping you to understand what is going on in the
server with authenticating/authorizing HTTP requests. You can switch the level of the following loggers:

    <Logger name="icCube.http.request" level="info"/>
    <Logger name="icCube.authorization" level="info"/>

from `info` to `debug` as following:

    <Logger name="icCube.http.request" level="debug"/>
    <Logger name="icCube.authorization" level="debug"/>

This way, you can review the HTTP headers received by icCube and the activities related to the both the
authentication and authorization. Search for example for `[auth]` in the console.

### Source Code

The [ic3-demo-embedded-react](https://github.com/ic3-software/ic3-demo-embedded-react) GitHub repository contains
the source code of a live example that demonstrates how to embed dashboards (and server administration) into an
existing React application.

You can search for the `withCustomHeaders` in the source code (it is switched off by default) and how it is used
to configure accordingly the URLs and/or the icCube API calls.

### Reverse proxy : example implementation

This is an example implementation of a reverse proxy in TypeScript. 

```ts
import {createProxyMiddleware} from 'http-proxy-middleware';
import {RequestHandler} from "express";
import express = require('express');


const app = express();


/**
 * Reverse proxy to icCube. Replace http://localhost:8282/ with the URL where your instance is hosted.
 */
const apiProxy = createProxyMiddleware({
    target: "http://localhost:8282/",
    changeOrigin: false,
    onProxyReq(proxyReq, req: any, _res) {

        // Be sure to remove auth headers. If not, an attacker could spoof these headers and gain access.
        proxyReq.removeHeader("x-ic3-user-id")
        proxyReq.removeHeader("x-ic3-role");
        proxyReq.removeHeader("x-ic3-locale");

        if (req.authInfo) {

            proxyReq.setHeader("x-ic3-user-id", req.authInfo.userId);
            proxyReq.setHeader("x-ic3-role", req.authInfo.userRole);
            proxyReq.setHeader("x-ic3-locale", req.authInfo.userLocale);

        }

    }
});

/**
 * When loading the reporting files (.js, CSS and HTML), the request does not contain the JWT token. IcCube then
 * uses the `publicAssetsRole` defined in the passthrough filter.
 */
function pathIsPublicPath(path: string) {

    const isAuthorizedPath =
        (path.startsWith("/doc") && !path.startsWith("/doc/ic3-reporting/app"))
        ||
        path.startsWith("/report/ic3-reporting/app-local")
        ||
        path.startsWith("/ux-api")
        ||
        path.startsWith("/api")
        ||
        path.startsWith("/gvi")
    ;

    console.log(path, isAuthorizedPath)

    return !isAuthorizedPath;

}

/**
 * This middleware checks if the requested resource is public.
 * If it is private, then it adds user info to the request from the JWT token.
 */
const authMiddleware: RequestHandler = (req, res, next) => {

    delete (req as any)['authInfo'];

    if (pathIsPublicPath(req.path)) {

        /*
        Public paths do not require authorization
         */

        next();

    } else if (req.headers.authorization) {

        /*
        Process here the authorization header from the JWT-token.
         */

        const userId = "1";  // In production, get this from the JWT-token.
        const userRole = "administrator";  // In production, get this from the JWT-token.
        const userLocale = "en_US";  // In production, get this from the JWT-token.

        (req as any)['authInfo'] = {
            userId, userRole, userLocale
        }

        next();


    } else {

        /*
        Unauthorized
         */

        res.sendStatus(401);

    }

}

app.use('/icCube', authMiddleware, apiProxy);

app.listen(4000);

```

### Contact Us

Please do not hesitate to [contact us](https://www.iccube.com/contact-us/) for more details.

_