/*
 * Copyright 2014 - 2021 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.excel.datasource;

import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.IOlapBuilderTableRowReader;
import crazydev.iccube.builder.errors.OlapBuilderErrorManager;
import crazydev.iccube.builder.excel.errors.OlapExcelBuilderErrorCode;
import crazydev.iccube.builder.factory.schema.IOlapBuilderJaxbListener;
import crazydev.iccube.builder.goodies.headerdef.column.OlapBuilderColumnDefinitionParser;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderDataSource;
import crazydev.iccube.builder.model.def.IOlapBuilderDataTableDef;
import crazydev.iccube.builder.model.impl.OlapBuilderDataColumn;
import crazydev.iccube.builder.model.impl.table.OlapBuilderBaseDataTable;
import crazydev.iccube.builder.model.validation.OlapBuilderValidator;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlRootElement;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Row;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@XmlRootElement(name = "excelDataTable")
public class OlapBuilderExcelDataTable extends OlapBuilderBaseDataTable<OlapBuilderExcelConnection> implements IOlapBuilderJaxbListener
{
    public static final CdProperty SHEET_NAME = new CdReadProperty(OlapBuilderExcelDataTable.class, "sheetName");

    public static final CdProperty START_ROW_NUMBER = new CdReadWriteProperty(OlapBuilderExcelDataTable.class, "headerRowNumber", false);

    public static final CdProperty TRIMSTRINGS = new CdReadWriteProperty(OlapBuilderExcelDataTable.class, "trimStrings");

    public static final CdProperty ADDROWNUMBER = new CdReadWriteProperty(OlapBuilderExcelDataTable.class, "addRowNumber");

    public static final String COL_ROWNUM_NAME = "#RowNumber";

    @XmlAttribute(required = false)
    private String sheetName;

    @XmlAttribute(required = false)
    private boolean trimStrings;

    @Nullable
    @XmlAttribute(required = false)
    private Integer headerRowNumber;

    @XmlAttribute(required = false)
    private boolean addRowNumber;

    public OlapBuilderExcelDataTable()
    {
    }

    public OlapBuilderExcelDataTable(String sheetName)
    {
        super(sheetName);

        this.sheetName = sheetName;
    }

    public OlapBuilderExcelDataTable(String sheetName, boolean trimStrings, boolean addRowNumber, List<IOlapBuilderDataColumnDef> columns)
    {
        super(sheetName, columns);

        this.sheetName = sheetName;
        this.trimStrings = trimStrings;
        this.addRowNumber = addRowNumber;
    }

    @Override
    public boolean isNameReadOnly()
    {
        return false;
    }

    @Override
    public void beforeMarshal()
    {

    }

    @Override
    public void afterUnmarshal()
    {
        if (sheetName == null)
        {
            sheetName = getName();
        }
    }

    @Override
    public String getInternalFriendlyTypeName()
    {
        return "Excel file Table";
    }

    @Nullable
    public Integer getHeaderRowNumber()
    {
        return headerRowNumber;
    }

    /**
     * UI support.
     * <p/>
     * Discover the whole list of available columns for this data-table; each column being setup as selected.
     * This data-table list of column is not updated.
     */
    @Override
    protected List<? extends IOlapBuilderDataColumnDef> doDiscoverAllColumns(OlapRuntimeContext context, OlapBuilderExcelConnection openedConnection, OlapBuilderErrorManager errorManager)
    {
        final List<IOlapBuilderDataColumnDef> columns = new ArrayList<>();
        final OlapBuilderExcelConnection.Accessor accessor = openedConnection.getSheetAccessor(getSheetName(), getActualStartRows());
        if (accessor.isEmptySheet())
        {
            errorManager.addError(OlapExcelBuilderErrorCode.EXCEL_SHEET_IS_EMPTY, sheetName);
            return columns;
        }
        else
        {
            final Row row = accessor.getHeaderRow();
            final int cellCount = row.getPhysicalNumberOfCells();
            if (cellCount == 0)
            {
                errorManager.addError(OlapExcelBuilderErrorCode.EXCEL_SHEET_IS_EMPTY, sheetName);
                return columns;
            }

            if (addRowNumber)
            {
                columns.add(new OlapBuilderDataColumn(OlapBuilderInputType.INTEGER, "rownum", COL_ROWNUM_NAME));
            }

            for (int col = 0; col < cellCount; col++)
            {
                //prevent getting null
                final Cell cell = row.getCell(col, Row.MissingCellPolicy.CREATE_NULL_AS_BLANK);
                final CellType cellType = cell.getCellType();
                switch (cellType)
                {
                    case BLANK:
                    {
                        break;
                    }
                    case NUMERIC:
                    case STRING:
                    {
                        final String columnName = getColumnNameFromCell(cell);
                        //case when we have empty sheet. But there is one (empty) cell inn the first row
                        if (cellCount == 1 && StringUtils.isBlank(columnName))
                        {
                            errorManager.addError(OlapExcelBuilderErrorCode.EXCEL_SHEET_IS_EMPTY, sheetName);
                            return columns;
                        }

                        if (StringUtils.isBlank(columnName))
                        {
                            errorManager.addError(OlapExcelBuilderErrorCode.EXCEL_COLUMN_NAME_IS_EMPTY, col, sheetName);
                            break;
                        }

                        final OlapBuilderDataColumn column = OlapBuilderColumnDefinitionParser.getInstance().parse(columnName);
                        final OlapBuilderExcelType columnType = getColumnType(accessor, col);

                        // don't change the type if it's coming from the header (String is the default type)
                        if (!columnName.contains("<Type.") && column.getType() == OlapBuilderInputType.STRING)
                        {
                            // ok we didn't define the type in the column, so let's put something
                            column.setType(columnType.getType());
                        }
                        column.setTableType(columnType.toString());
                        columns.add(column);
                        break;
                    }
                    default:
                    {
                        errorManager.addError(OlapExcelBuilderErrorCode.EXCEL_INCORRECT_CELL_TYPE_FOR_COLUMN, sheetName, col);
                        break;
                    }
                }
            }

        }

        return columns;
    }

    protected final String getColumnNameFromCell(Cell cell)
    {
        final String colName = cell.getCellType() == CellType.STRING ? cell.getStringCellValue() : String.valueOf(cell.getNumericCellValue());
        return colName.replace("\n", " ");
    }

    @Override
    public List<? extends IOlapBuilderDataColumnDef> doDiscoverAllColumnsForRefresh(OlapRuntimeContext context, OlapBuilderExcelConnection connection, OlapBuilderErrorManager errorManager)
    {
        return discoverAllColumns(context, connection, errorManager);
    }

    private OlapBuilderExcelType getColumnType(OlapBuilderExcelConnection.Accessor sheet, int col)
    {
        // just do it for the first one
        Row row = sheet.getFirstDataRow();
        if (row == null)
        {
            return OlapBuilderExcelType.EXCEL_UNDEFINED;
        }

        final Cell cell = row.getCell(col, Row.MissingCellPolicy.CREATE_NULL_AS_BLANK);
        final OlapBuilderExcelType cellType = getCellType(cell);
        switch (cellType)
        {
            case EXCEL_NUMERIC:
                return OlapBuilderExcelType.EXCEL_NUMERIC;
            case EXCEL_DATE:
                return OlapBuilderExcelType.EXCEL_DATE;
            case EXCEL_STRING:
                return OlapBuilderExcelType.EXCEL_STRING;
            case EXCEL_BOOLEAN:
                return OlapBuilderExcelType.EXCEL_BOOLEAN;
            case EXCEL_BLANK:
            case EXCEL_UNDEFINED:
                return OlapBuilderExcelType.EXCEL_UNDEFINED;
        }
        return OlapBuilderExcelType.EXCEL_UNDEFINED;
    }

//    public void buildColumns(OlapBuilderExcelConnection connection, OlapBuilderErrorManager errorManager)
//    {
//        final List<? extends IOlapBuilderDataColumnDef> columns = discoverAllColumns(, connection, errorManager);
//
//        for (IOlapBuilderDataColumnDef column : columns)
//        {
//            addColumn(column);
//        }
//    }

    @Override
    protected IOlapBuilderTableRowReader<OlapBuilderExcelConnection> doCreateFullTableRowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount)
    {
        return new OlapBuilderExcelDataTableRowReader(context, connectionPool, maxRowCount, this, getActualStartRows());
    }

    /**
     * Used by Google spreadsheet.
     */
    public OlapBuilderExcelDataTableRowReader createFullTableRowReaderForGoogle(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount, OlapBuilderExcelConnection connection)
    {
        return new OlapBuilderExcelDataTableRowReader(context, connectionPool, maxRowCount, this, connection, getActualStartRows());
    }

    private int getActualStartRows()
    {
        return headerRowNumber == null ? 0 : Math.max(0, headerRowNumber - 1);
    }

    protected Map<String, Integer> createMapping(final Row row)
    {
        final Map<String, Integer> columnMapping = new HashMap<>();
        final int cellCount = row == null ? 0 : row.getPhysicalNumberOfCells();

        for (int col = 0; col < cellCount; col++)
        {
            final Cell cell = row.getCell(col, Row.MissingCellPolicy.CREATE_NULL_AS_BLANK);
            final String columnName = getColumnNameFromCell(cell);
            final IOlapBuilderDataColumnDef column = OlapBuilderColumnDefinitionParser.getInstance().parse(columnName);
            IOlapBuilderDataColumnDef columnDef = lookupSelectedColumn(column.getName());
            if (columnDef != null)
            {
                columnMapping.put(columnDef.getName(), col);
            }
        }

        return columnMapping;
    }

    public static OlapBuilderExcelType getCellType(Cell cell)
    {
        return getICellType(cell, cell.getCellType(), false);
    }

    public static boolean isDateCell(Cell cell)
    {
        try
        {
            return DateUtil.isCellDateFormatted(cell);
        }
        catch (RuntimeException ignored)
        {
            // special case
            return false;
        }
    }

    private static OlapBuilderExcelType getICellType(Cell cell, CellType cellType, boolean rec)
    {
        switch (cellType)
        {
            case NUMERIC:
            {
                boolean isDate = isDateCell(cell);
                return isDate ? OlapBuilderExcelType.EXCEL_DATE : OlapBuilderExcelType.EXCEL_NUMERIC;
            }
            case BOOLEAN:
            {
                return OlapBuilderExcelType.EXCEL_BOOLEAN;
            }
            case STRING:
            {
                return OlapBuilderExcelType.EXCEL_STRING;
            }
            case FORMULA:
            {
                // avoid infinite looping
                if (rec)
                {
                    return OlapBuilderExcelType.EXCEL_UNDEFINED;
                }
                else
                {
                    return getICellType(cell, cell.getCachedFormulaResultType(), true);
                }
            }
            case BLANK:
            {
                return OlapBuilderExcelType.EXCEL_BLANK;
            }
            default:
            {
                return OlapBuilderExcelType.EXCEL_UNDEFINED;
            }
        }
    }

    @Override
    public OlapBuilderValidator<IOlapBuilderDataSource<OlapBuilderExcelConnection>, IOlapBuilderDataTableDef<OlapBuilderExcelConnection>> getValidator()
    {
        return new OlapBuilderExcelDataTableValidator();
    }

    public boolean getTrimStrings()
    {
        return trimStrings;
    }

    @Override
    @Nullable
    protected IOlapBuilderTableRowReader createRowReaderForTableCacheRead(OlapBuilderContext context)
    {
        return null  /* not much sense for us I guess */;
    }

    @Override
    protected IOlapBuilderTableRowReader decorateRowReaderForTableCache(OlapBuilderContext context, IOlapBuilderTableRowReader reader)
    {
        return reader /* not much sense for us I guess */;
    }

    public String getSheetName()
    {
        return sheetName;
    }
}
